/*
 * Copyright (c) 2014-2019 Meltytech, LLC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
 
// KEYFRAMES to b added.

import QtQuick 2.1
import QtQuick.Controls 1.1
import QtQuick.Layouts 1.1
import Shotcut.Controls 1.0
import QtQuick.Dialogs  1.0

Item {
	Item {
		id:	fp1											// Formal Parameter 1
		property string	pname		: 'fp1'				// Formal Parameter name
		property string ptext		: 'Amount'			// Formal Parameter title shown on filter
		property real   dflt		:  5				// Default value
		property var	mnmx		: [0, 60]			// Scale [min, max]
		property real	slen		: mnmx[1] - mnmx[0]	// Scale length
		property var    sme			: [0.0, 0.5, 1.0]	// min, average, max values
		property var    scale		: 1					// Scale is NOT a percentage
		property var    stepsz		: 1					// stepSize (relative to Scale)
		property var    decplaces	: 0					// decimal places to display
	}

    property bool	blockUpdate		: true
	
    property var	startValues		: [fp1.sme[0]]		// Amount
    property var	middleValues	: [fp1.sme[1]]
    property var	endValues		: [fp1.sme[2]]
	
    width : 350
    height: 100
	
    Component.onCompleted: {
        if (filter.isNew) {
 			filter.set('resource', filter.path + 'Jitter.html')
            filter.set('fp1'	 , fp1.dflt)
            filter.savePreset(preset.parameters)
		} else {
            initSimpleAnimation()
		}
        setControls();
		setKeyframedControls();
//        filter.set('in' , producer.in)	//Workaround for an advanced key frame issue when the filter is trimmed
//        filter.set('out', producer.out)	//Workaround for an advanced key frame issue when the filter is trimmed
    }
	
	function initSimpleAnimation() {	// Strength
		middleValues 	= [
						   filter.getDouble('fp1', filter.animateIn)
						  ]
        if (filter.animateIn > 0) {
            startValues = [
						   filter.getDouble('fp1', 0)
						  ]
        }
        if (filter.animateOut > 0) {
            endValues 	= [
						   filter.getDouble('fp1', filter.duration - 1)
						  ]
        }
    }

    function getPosition() {
        return Math.max(producer.position - (filter.in - producer.in), 0)
    }

    function setKeyframedControls() {	// Key-framed
        var position	= getPosition()
        blockUpdate		= true
		
        fp1Slider.value = fp1Slider.minimumValue + filter.getDouble('fp1', position) * fp1.scale
		
        blockUpdate	= false
        fp1Slider.enabled	= position <= 0                         ||
								(position >= (
											  filter.animateIn - 1) && position <= (filter.duration - filter.animateOut)
											 )                      ||
								 position >= (filter.duration - 1
								)
    }	
	
    function setControls() {		// Non-Key-framed
        var position        = getPosition()
    }

    function updateFilter(parameter, value, position, button) {
        if (blockUpdate) return
        var index = preset.parameters.indexOf(parameter)
        if (position !== null) {
            if (position <= 0 && filter.animateIn > 0)
                startValues[index]	= value
            else if (position >= filter.duration - 1 && filter.animateOut > 0)
                endValues[index]	= value
            else
                middleValues[index]	= value
        }

        if (filter.animateIn > 0 || filter.animateOut > 0) {
            filter.resetProperty(parameter)
           button.checked = false
            if (filter.animateIn > 0) {
                filter.set(parameter,  startValues[index], 0)
                filter.set(parameter, middleValues[index], filter.animateIn - 1)
			}
            if (filter.animateOut > 0) {
                filter.set(parameter, middleValues[index], filter.duration - filter.animateOut)
                filter.set(parameter,    endValues[index], filter.duration - 1)
            }
        } else if (!button.checked) {
            filter.resetProperty(parameter)
            filter.set			(parameter, middleValues[index])
        } else if (position !== null) {
            filter.set			(parameter, value, position)
        }
   }

    function onKeyframesButtonClicked(checked, parameter, value) {
        if (checked) {
            blockUpdate = true
            fp1Slider.enabled = true
            if (filter.animateIn > 0 || filter.animateOut > 0) {
                filter.resetProperty('fp1')				
                filter.animateIn = filter.animateOut = 0
            } else {
                filter.clearSimpleAnimation(parameter)
            }
            blockUpdate = false
            filter.set(parameter, value, getPosition())
        } else {
            filter.resetProperty(parameter)
            filter.set(parameter, value)
        }
    }

    GridLayout {
        columns			: 4
        anchors.fill	: parent
        anchors.margins	: 8

        Label {
            text: qsTr('Preset')
            Layout.alignment: Qt.AlignRight
        }

       Preset {
            id					: preset
            Layout.columnSpan	: parent.columns - 1
            parameters			: ['fp1']	
            onPresetSelected	: {
									setControls()
									setKeyframedControls()
									initSimpleAnimation()
			}
        }

        Label {
            text				: qsTr(fp1.ptext)		// Amount (Key-framed)
            Layout.alignment	: Qt.AlignRight
        }
        SliderSpinner {
            id					: fp1Slider
			minimumValue		: fp1.mnmx[0]
            maximumValue		: fp1.mnmx[1]
			stepSize			: fp1.stepsz
			decimals			: fp1.decplaces
            suffix				: ''
            value				: minimumValue + filter.getDouble('fp1') * fp1.scale
            onValueChanged		: updateFilter('fp1', value / fp1.scale, getPosition(), fp1KeyframesButton)
        }
        UndoButton {
            onClicked			: fp1Slider.value = fp1Slider.minimumValue + fp1.dflt * fp1.scale
        }
        KeyframesButton {
            id					: fp1KeyframesButton
            checked				: filter.animateIn <= 0 && filter.animateOut <= 0 && filter.keyframeCount('fp1') > 0
            onToggled			: onKeyframesButtonClicked(checked, 'fp1', fp1Slider.value / fp1.scale)
        }

        Label {
            text					: qsTr('Description')		// Link to filter description
			Layout.alignment		: Qt.AlignRight
       }
		Button {
            id						: description
            Layout.alignment		: Qt.AlignLeft
			text					: "Details"
			onClicked				: Qt.openUrlExternally('http://elusien.co.uk/shotcut/filters/jitter')
        }		
        Item { Layout.fillWidth		: true }		
	}

    function updateSimpleAnimation() {
        updateFilter('fp1', fp1Slider.minimumValue + fp1Slider.value / fp1.scale, getPosition(), fp1KeyframesButton)
	}

    Connections {
        target				: filter
        onInChanged			: updateSimpleAnimation()
        onOutChanged		: updateSimpleAnimation()
        onAnimateInChanged	: updateSimpleAnimation()
        onAnimateOutChanged	: updateSimpleAnimation()
    }

    Connections {
        target				: producer
        onPositionChanged	: setKeyframedControls()
    }
}
